<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';

// Set JSON header
header('Content-Type: application/json');

// Check if user has permission to manage supplier orders
$supplierPermissions = [
    'manage_suppliers', 'add_supplier', 'edit_supplier'
];

if (!hasAnyPermission($supplierPermissions) && !$isSuperAdmin) {
    echo json_encode(['success' => false, 'message' => 'Access denied. You do not have permission to update order status.']);
    exit;
}

// Get JSON input
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data || !isset($data['order_id']) || !isset($data['status'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid request data']);
    exit;
}

$order_id = intval($data['order_id']);
$new_status = trim($data['status']);

// Validate status
$valid_statuses = ['pending', 'approved', 'completed', 'cancelled'];
if (!in_array($new_status, $valid_statuses)) {
    echo json_encode(['success' => false, 'message' => 'Invalid status value']);
    exit;
}

try {
    $pdo->beginTransaction();
    
    // Check if order exists and get current status
    $stmt = $pdo->prepare("
        SELECT id, status, supplier_id 
        FROM supplier_orders 
        WHERE id = ?
    ");
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        throw new Exception('Order not found');
    }
    
    $old_status = $order['status'];
    
    // Check if status is actually changing
    if ($old_status === $new_status) {
        echo json_encode(['success' => false, 'message' => 'Order already has this status']);
        exit;
    }
    
    // Update order status
    $stmt = $pdo->prepare("
        UPDATE supplier_orders 
        SET status = ?, updated_at = NOW() 
        WHERE id = ?
    ");
    $stmt->execute([$new_status, $order_id]);
    
    // If order is completed, update all pending items to received
    if ($new_status === 'completed') {
        $stmt = $pdo->prepare("
            UPDATE supplier_order_items 
            SET status = 'received', received_quantity = quantity 
            WHERE order_id = ? AND status IN ('pending', 'partial')
        ");
        $stmt->execute([$order_id]);
    }
    
    // If order is cancelled, update all items to cancelled
    if ($new_status === 'cancelled') {
        $stmt = $pdo->prepare("
            UPDATE supplier_order_items 
            SET status = 'cancelled' 
            WHERE order_id = ? AND status != 'received'
        ");
        $stmt->execute([$order_id]);
    }
    
    // Get supplier name for logging
    $stmt = $pdo->prepare("
        SELECT s.name as supplier_name 
        FROM suppliers s 
        JOIN supplier_orders so ON s.id = so.supplier_id 
        WHERE so.id = ?
    ");
    $stmt->execute([$order_id]);
    $supplier = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Log the action
    logSystemAction(
        'supplier_order_status_updated',
        "Updated order #{$order_id} status from '{$old_status}' to '{$new_status}' for supplier: {$supplier['supplier_name']}"
    );
    
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Order status updated successfully',
        'old_status' => $old_status,
        'new_status' => $new_status
    ]);
    
} catch (Exception $e) {
    $pdo->rollBack();
    error_log("Error updating order status: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'An error occurred while updating order status: ' . $e->getMessage()]);
}
?> 